#include "MTTXMPViewer.h"
#include "MTUtilities.h"
#include "MTFileList.h"
#include "IFileStream.h"
#include "IErrors.h"
#include "MTExceptions.h"
#include "MTApplication.h"

/****** TXMP Format Info ******************************************************
 *	
 *	0:	AAAARRRRGGGGBBBB
 *	1:	ARRRRRGGGGGBBBBB
 *	2:	ARRRRRGGGGGBBBBB
 *	8:	AAAAAAAARRRRRRRRGGGGGGGGBBBBBBBB
 *	9:	Compressed 4x4 block method
 *		RRRRRGGGGGGBBBBB RRRRRGGGGGGBBBBB XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX
 *		This method stores the texture as several 4x4 blocks. Each block starts
 *		with two colors. Then follows 32 bits of pixel information. The pixel
 *		block is laid out like this (two bits per pixel):
 *		
 *		03 02 01 00
 *		07 06 05 04
 *		11 10 09 08
 *		15 14 13 12
 *		
 *		The first and second colors of the palette used to decode the block are
 *		the first and second colors read for the block. The third and fourth
 *		colors are the two colors in-between the first and second.
 *		
 *****************************************************************************/

struct TXMPStructure
{
	UInt32	id;
	UInt32	unknown1;
	char	name[0x80];
	UInt32	unknown2;
	UInt16	width;
	UInt16	height;
	UInt32	format;
	UInt32	unknown4;
	UInt32	unknown5;
	UInt32	unknown6;
	UInt32	dataOffset;
	UInt32	unknown7;
};
		
MTInfoWindowDataTemplate	MTTXMPViewer::kInfoWindowTemplate =
{
	12,
	{
		{	"ID" },
		{	"Unknown" },
		{	"Name" },
		{	"Unknown" },
		{	"Width" },
		{	"Height" },
		{	"Format" },
		{	"Unknown" },
		{	"Unknown" },
		{	"Unknown" },
		{	"Data Offset" },
		{	"Unknown" }
	}
};

MTTXMPHandler::MTTXMPHandler()
{
	
}

MTTXMPHandler::~MTTXMPHandler()
{
	
}

void MTTXMPHandler::ParseData(UInt8 * buf, UInt32 size)
{
	TXMPStructure	* structure = (TXMPStructure *)buf;
	
	if(size < sizeof(TXMPStructure))
		throw MTUnsupportedException("Unsupported data structure");
	
	SWAP32(structure->id);
	SWAP32(structure->unknown1);
	SWAP32(structure->unknown2);
	SWAP32(structure->format);
	SWAP32(structure->unknown4);
	SWAP32(structure->unknown5);
	SWAP32(structure->unknown6);
	SWAP32(structure->unknown7);
	SWAP32(structure->dataOffset);
	SWAP16(structure->width);
	SWAP16(structure->height);
	
	structure->name[0x80 - 1] = 0;	// force termination
	
	id =			structure->id;
	unknown1 =		structure->unknown1;
	name =			structure->name;
	unknown2 =		structure->unknown2;
	width =			structure->width;
	height =		structure->height;
	format =		structure->format;
	unknown4 =		structure->unknown4;
	unknown5 =		structure->unknown5;
	unknown6 =		structure->unknown6;
	dataOffset =	structure->dataOffset;
	unknown7 =		structure->unknown7;
	
	switch(format)
	{
		case 0:
			depth = 16;
			dataSize = width * height * 2;
			break;
		
		case 1:
			depth = 16;
			dataSize = width * height * 2;
			break;
		
		case 2:
			depth = 16;
			dataSize = width * height * 2;
			break;
		
		case 8:
			depth = 32;
			dataSize = width * height * 4;
			break;
		
		case 9:
			depth = 16;
			dataSize = width * height / 2;
			break;
		
		default:
			throw MTUnsupportedException("Unsupported TXMP format");
			
			depth = 16;
			dataSize = width * height * 2;
			break;
	}
}

void MTTXMPHandler::CopyFromGWorld(GWorldPtr theWorld)
{
	UInt32	worldWidth = theWorld->portRect.right - theWorld->portRect.left;
	UInt32	worldHeight = theWorld->portRect.bottom - theWorld->portRect.top;
	UInt8	* outBuffer;
	
	if((worldWidth != width) || (worldHeight != height))
		throw MTException("Picture size doesn't match");
	
	outBuffer = (UInt8 *)NewPtr(dataSize);
	if(!outBuffer)
		throw MTMemoryException("Out of memory");
	
	{
		UInt32			* basePtr;
		UInt32			rowBytes;
		UInt8			* outTraverse;
		UInt32			* inTraverse;
		
		LockPixels(GetGWorldPixMap(theWorld));
		
		basePtr = (UInt32 *)GetPixBaseAddr(GetGWorldPixMap(theWorld));
		rowBytes = GetPixRowBytes(GetGWorldPixMap(theWorld));
		outTraverse = outBuffer;
		
		switch(format)
		{
			case 0:	// AAAARRRRGGGGBBBB
			{
				for(UInt32 y = 0; y < height; y++)
				{
					inTraverse = (UInt32 *)(((UInt8 *)basePtr) + (rowBytes * y));
					
					for(UInt32 x = 0; x < width; x++)
					{
						UInt32	data;
						UInt32	red, green, blue;
						
						data = inTraverse[x];
						
						red =	((data >> 16) & 0xFF) >> 4;
						green =	((data >>  8) & 0xFF) >> 4;
						blue =	((data >>  0) & 0xFF) >> 4;
						
						data = (red << 8) | (green << 4) | (blue << 0);
						
						*((UInt16 *)outTraverse)++ = SWAPRET16(data);
					}
				}
			}
			break;
			
			case 1:	// ARRRRRGGGGGBBBBB
			case 2:	// ARRRRRGGGGGBBBBB
			{
				for(UInt32 y = 0; y < height; y++)
				{
					inTraverse = (UInt32 *)(((UInt8 *)basePtr) + (rowBytes * y));
					
					for(UInt32 x = 0; x < width; x++)
					{
						UInt32	data;
						
						data = Convert32BitPixelTo16Bit(inTraverse[x]);
						
						*((UInt16 *)outTraverse)++ = SWAPRET16(data);
					}
				}
			}
			break;
			
			case 8:	// AAAAAAAARRRRRRRRGGGGGGGGBBBBBBBB
			{
				for(UInt32 y = 0; y < height; y++)
				{
					inTraverse = (UInt32 *)(((UInt8 *)basePtr) + (rowBytes * y));
					
					for(UInt32 x = 0; x < width; x++)
					{
						UInt32	data;
						
						data = inTraverse[x];
						
						*((UInt32 *)outTraverse)++ = SWAPRET32(data);
					}
				}
			}
			break;
			
			case 9:	// 4x4 square compression
			{
				for(UInt32 y = 0; y < height; y += 4)
				{
					for(UInt32 x = 0; x < width; x += 4)
					{
						UInt32	highColor =	0x00000000;
						UInt32	midHighColor;
						UInt32	midLowColor;
						UInt32	lowColor =	0xFFFFFFFF;
						UInt32	data;
						
						for(UInt32 y2 = 0; y2 < 4; y2++)
						{
							inTraverse = (UInt32 *)(((UInt8 *)basePtr) + (rowBytes * (y + y2)));
							
							for(UInt32 x2 = 0; x2 < 4; x2++)
							{
								data = inTraverse[x + x2];
								
								if(data > highColor)
									highColor = data;
								if(data < lowColor)
									lowColor = data;
							}
						}
						
						data = Convert32BitPixelTo56516Bit(highColor);
						*((UInt16 *)outTraverse)++ = SWAPRET16(data);
						
						data = Convert32BitPixelTo56516Bit(lowColor);
						*((UInt16 *)outTraverse)++ = SWAPRET16(data);
						
						{
							UInt32	red1,	green1,	blue1;
							UInt32	red2,	green2,	blue2;
							UInt32	red,	green,	blue;
							
							red1 =		(highColor >> 16) & 0xFF;
							green1 =	(highColor >>  8) & 0xFF;
							blue1 =		(highColor >>  0) & 0xFF;
							
							red2 =		(lowColor >>  16) & 0xFF;
							green2 =	(lowColor >>   8) & 0xFF;
							blue2 =		(lowColor >>   0) & 0xFF;
							
							red =		Interpolate(33, 0, 100, red1, red2);
							green =		Interpolate(33, 0, 100, green1, green2);
							blue =		Interpolate(33, 0, 100, blue1, blue2);
							
							midHighColor =	((red <<  16) & 0x00FF0000) |
											((green << 8) & 0x0000FF00) |
											((blue <<  0) & 0x000000FF);
							
							red =		Interpolate(66, 0, 100, red1, red2);
							green =		Interpolate(66, 0, 100, green1, green2);
							blue =		Interpolate(66, 0, 100, blue1, blue2);
							
							midLowColor =	((red <<  16) & 0x00FF0000) |
											((green << 8) & 0x0000FF00) |
											((blue <<  0) & 0x000000FF);
						}
						
						for(UInt32 y2 = 0; y2 < 4; y2++)
						{
							inTraverse = (UInt32 *)(((UInt8 *)basePtr) + (rowBytes * (y + y2)));
							
							UInt8	packet = 0;
							
							for(SInt32 x2 = 3; x2 >= 0; x2--)
							{
								UInt32	score1, score2, score3, score4, minScore;
								
								data = inTraverse[x + x2];
								
								score1 = CalculateColorDifference(data, highColor);
								score2 = CalculateColorDifference(data, lowColor);
								score3 = CalculateColorDifference(data, midHighColor);
								score4 = CalculateColorDifference(data, midLowColor);
								
								minScore = score1;
								if(score2 < minScore)
									minScore = score2;
								if(score3 < minScore)
									minScore = score3;
								if(score4 < minScore)
									minScore = score4;
								
								packet <<= 2;
								
								if(minScore == score1)
									packet |= 0;
								else if(minScore == score2)
									packet |= 1;
								else if(minScore == score3)
									packet |= 2;
								else if(minScore == score4)
									packet |= 3;
							}
							
							*outTraverse++ = packet;
						}
					}
				}
			}
			break;
		}
	}
	
	{
		IFileStream	* outStream = fileList->theFile->OpenDataFile();
		
		outStream->SetPosition(dataOffset);
		outStream->WriteBuffer(outBuffer, dataSize);
		outStream->CloseFile();
		
		delete outStream;
	}
	
	DisposePtr((Ptr)outBuffer);
}

void MTTXMPHandler::DrawToBuffer(UInt8 * pixBaseAddr, UInt32 rowBytes, UInt32 inDepth)
{
	UInt8		* buf;
	
	buf = fileList->theFile->LoadSegment(dataOffset, dataSize);
	
	switch(format)
	{
		case 0:	// AAAARRRRGGGGBBBB
		{
			if(inDepth == 16)
			{
				UInt16	* fromPtr = (UInt16 *)buf;
				
				for(UInt32 y = 0; y < height; y++)
				{
					UInt16	* linePtr = (UInt16 *)(pixBaseAddr + (y * rowBytes));
					
					for(UInt32 x = 0; x < width; x++)
					{
						UInt16	data = *fromPtr++;
						UInt16	red, green, blue, alpha;
						
						// ARGB
						// GBAR
						
						red =	((data >>  0) & 0x000F) << 1;
						green =	((data >> 12) & 0x000F) << 1;
						blue =	((data >>  8) & 0x000F) << 1;
						alpha =	((data >>  4) & 0x000F) << 1;
						
						*linePtr++ = (red << 10) | (green << 5) | (blue << 0);
					}
				}
			}
			else
			{
				if(inDepth == 32)
				{
					UInt16	* fromPtr = (UInt16 *)buf;
					
					for(UInt32 y = 0; y < height; y++)
					{
						UInt32	* linePtr = (UInt32 *)(pixBaseAddr + (y * rowBytes));
						
						for(UInt32 x = 0; x < width; x++)
						{
							UInt16	data = *fromPtr++;
							UInt16	red, green, blue, alpha;
							
							// ARGB
							// GBAR
							
							red =	((data >>  0) & 0x000F) << 1;
							green =	((data >> 12) & 0x000F) << 1;
							blue =	((data >>  8) & 0x000F) << 1;
							alpha =	((data >>  4) & 0x000F) << 1;
							
							*linePtr++ = Convert16BitPixelTo32Bit((red << 10) | (green << 5) | (blue << 0));
						}
					}
				}
			}
		}
		break;
		
		case 1:	// ARRRRRGGGGGBBBBB
		case 2:	// ARRRRRGGGGGBBBBB
		{
			if(inDepth == 16)
			{
				UInt16	* fromPtr = (UInt16 *)buf;
				
				for(UInt32 y = 0; y < height; y++)
				{
					UInt16	* linePtr = (UInt16 *)(pixBaseAddr + (y * rowBytes));
					
					for(UInt32 x = 0; x < width; x++)
					{
						UInt16	data = *fromPtr++;
						
						SWAP16(data);
						
						*linePtr++ = data;
					}
				}
			}
			else
			{
				if(inDepth == 32)
				{
					UInt16	* fromPtr = (UInt16 *)buf;
					
					for(UInt32 y = 0; y < height; y++)
					{
						UInt32	* linePtr = (UInt32 *)(pixBaseAddr + (y * rowBytes));
						
						for(UInt32 x = 0; x < width; x++)
						{
							UInt16	data = *fromPtr++;
							
							SWAP16(data);
							
							*linePtr++ = Convert16BitPixelTo32Bit(data);
						}
					}
				}
			}
		}
		break;
		
		case 8:	// AAAAAAAARRRRRRRRGGGGGGGGBBBBBBBB
		{
			if(inDepth == 32)
			{
				UInt32	* fromPtr = (UInt32 *)buf;
				
				for(UInt32 y = 0; y < height; y++)
				{
					UInt32	* linePtr = (UInt32 *)(pixBaseAddr + (y * rowBytes));
					
					for(UInt32 x = 0; x < width; x++)
					{
						UInt32	data = *fromPtr++;
						
						SWAP32(data);
						
						*linePtr++ = data;
					}
				}
			}
		}
		break;
		
		case 9:	// 4x4 square compression
		{
			if(inDepth == 16)
			{
				UInt16	* fromPtr = (UInt16 *)buf;
				
				for(UInt32 y = 0; y < height; y += 4)
				{
					UInt16	* linePtr1 = ((UInt16 *)(pixBaseAddr + ((y + 0) * rowBytes))) + 3;
					UInt16	* linePtr2 = ((UInt16 *)(pixBaseAddr + ((y + 1) * rowBytes))) + 3;
					UInt16	* linePtr3 = ((UInt16 *)(pixBaseAddr + ((y + 2) * rowBytes))) + 3;
					UInt16	* linePtr4 = ((UInt16 *)(pixBaseAddr + ((y + 3) * rowBytes))) + 3;
					
					for(UInt32 x = 0; x < width; x += 4)
					{
						UInt16	color00;
						UInt16	color01;
						UInt16	color10;
						UInt16	color11;
						UInt32	data;
						UInt32	shift;
						
						UInt16	red1, green1, blue1;
						UInt16	red2, green2, blue2;
						UInt16	red, green, blue;
						
						color00 = *fromPtr++;
						color01 = *fromPtr++;
						data = *((UInt32 *)fromPtr)++;
						
						SWAP16(color00);
						SWAP16(color01);
						
						// RRRRRGGGGGGBBBBB
						
						red1 =		((color00 >> 11) & 0x1F) >> 0;
						green1 =	((color00 >>  5) & 0x3F) >> 1;
						blue1 =		((color00 >>  0) & 0x1F) >> 0;
						color00 = ((red1 & 0x1F) << 10) | ((green1 & 0x1F) << 5) | ((blue1 & 0x1F) << 0);
						
						red2 =		((color01 >> 11) & 0x1F) >> 0;
						green2 =	((color01 >>  5) & 0x3F) >> 1;
						blue2 =		((color01 >>  0) & 0x1F) >> 0;
						color01 = ((red2 & 0x1F) << 10) | ((green2 & 0x1F) << 5) | ((blue2 & 0x1F) << 0);
						
						red =	Interpolate(33, 0, 100, red1, red2);
						green =	Interpolate(33, 0, 100, green1, green2);
						blue =	Interpolate(33, 0, 100, blue1, blue2);
						color10 = ((red & 0x1F) << 10) | ((green & 0x1F) << 5) | ((blue & 0x1F) << 0);
						
						red =	Interpolate(66, 0, 100, red1, red2);
						green =	Interpolate(66, 0, 100, green1, green2);
						blue =	Interpolate(66, 0, 100, blue1, blue2);
						color11 = ((red & 0x1F) << 10) | ((green & 0x1F) << 5) | ((blue & 0x1F) << 0);
						
						shift = 30;
						for(UInt32 i = 0; i < 4; i++)
						{
							switch((data >> shift) & 3)
							{
								case 0:	*linePtr1 = color00; break;
								case 1:	*linePtr1 = color01; break;
								case 2:	*linePtr1 = color10; break;
								case 3:	*linePtr1 = color11; break;
							}
							
							linePtr1--;
							shift -= 2;
						}
						
						linePtr1 += 8;
						
						for(UInt32 i = 0; i < 4; i++)
						{
							switch((data >> shift) & 3)
							{
								case 0:	*linePtr2 = color00; break;
								case 1:	*linePtr2 = color01; break;
								case 2:	*linePtr2 = color10; break;
								case 3:	*linePtr2 = color11; break;
							}
							
							linePtr2--;
							shift -= 2;
						}
						
						linePtr2 += 8;
						
						for(UInt32 i = 0; i < 4; i++)
						{
							switch((data >> shift) & 3)
							{
								case 0:	*linePtr3 = color00; break;
								case 1:	*linePtr3 = color01; break;
								case 2:	*linePtr3 = color10; break;
								case 3:	*linePtr3 = color11; break;
							}
							
							linePtr3--;
							shift -= 2;
						}
						
						linePtr3 += 8;
						
						for(UInt32 i = 0; i < 4; i++)
						{
							switch((data >> shift) & 3)
							{
								case 0:	*linePtr4 = color00; break;
								case 1:	*linePtr4 = color01; break;
								case 2:	*linePtr4 = color10; break;
								case 3:	*linePtr4 = color11; break;
							}
							
							linePtr4--;
							shift -= 2;
						}
						
						linePtr4 += 8;
					}
				}
			}
			else
			{
				if(inDepth == 32)
				{
					UInt16	* fromPtr = (UInt16 *)buf;
					
					for(UInt32 y = 0; y < height; y += 4)
					{
						UInt32	* linePtr1 = ((UInt32 *)(pixBaseAddr + ((y + 0) * rowBytes))) + 3;
						UInt32	* linePtr2 = ((UInt32 *)(pixBaseAddr + ((y + 1) * rowBytes))) + 3;
						UInt32	* linePtr3 = ((UInt32 *)(pixBaseAddr + ((y + 2) * rowBytes))) + 3;
						UInt32	* linePtr4 = ((UInt32 *)(pixBaseAddr + ((y + 3) * rowBytes))) + 3;
						
						for(UInt32 x = 0; x < width; x += 4)
						{
							UInt32	color00;
							UInt32	color01;
							UInt32	color10;
							UInt32	color11;
							UInt32	data;
							UInt32	shift;
							
							UInt16	red1, green1, blue1;
							UInt16	red2, green2, blue2;
							UInt16	red, green, blue;
							
							color00 = *fromPtr++;
							color01 = *fromPtr++;
							data = *((UInt32 *)fromPtr)++;
							
							SWAP16(color00);
							SWAP16(color01);
							
							// RRRRRGGGGGGBBBBB
							
							red1 =		((color00 >> 11) & 0x1F) >> 0;
							green1 =	((color00 >>  5) & 0x3F) >> 1;
							blue1 =		((color00 >>  0) & 0x1F) >> 0;
							color00 = ((red1 & 0x1F) << 10) | ((green1 & 0x1F) << 5) | ((blue1 & 0x1F) << 0);
							
							red2 =		((color01 >> 11) & 0x1F) >> 0;
							green2 =	((color01 >>  5) & 0x3F) >> 1;
							blue2 =		((color01 >>  0) & 0x1F) >> 0;
							color01 = ((red2 & 0x1F) << 10) | ((green2 & 0x1F) << 5) | ((blue2 & 0x1F) << 0);
							
							red =	Interpolate(33, 0, 100, red1, red2);
							green =	Interpolate(33, 0, 100, green1, green2);
							blue =	Interpolate(33, 0, 100, blue1, blue2);
							color10 = ((red & 0x1F) << 10) | ((green & 0x1F) << 5) | ((blue & 0x1F) << 0);
							
							red =	Interpolate(66, 0, 100, red1, red2);
							green =	Interpolate(66, 0, 100, green1, green2);
							blue =	Interpolate(66, 0, 100, blue1, blue2);
							color11 = ((red & 0x1F) << 10) | ((green & 0x1F) << 5) | ((blue & 0x1F) << 0);
							
							color00 = Convert16BitPixelTo32Bit(color00);
							color01 = Convert16BitPixelTo32Bit(color01);
							color10 = Convert16BitPixelTo32Bit(color10);
							color11 = Convert16BitPixelTo32Bit(color11);
							
							shift = 30;
							for(UInt32 i = 0; i < 4; i++)
							{
								switch((data >> shift) & 3)
								{
									case 0:	*linePtr1 = color00; break;
									case 1:	*linePtr1 = color01; break;
									case 2:	*linePtr1 = color10; break;
									case 3:	*linePtr1 = color11; break;
								}
								
								linePtr1--;
								shift -= 2;
							}
							
							linePtr1 += 8;
							
							for(UInt32 i = 0; i < 4; i++)
							{
								switch((data >> shift) & 3)
								{
									case 0:	*linePtr2 = color00; break;
									case 1:	*linePtr2 = color01; break;
									case 2:	*linePtr2 = color10; break;
									case 3:	*linePtr2 = color11; break;
								}
								
								linePtr2--;
								shift -= 2;
							}
							
							linePtr2 += 8;
							
							for(UInt32 i = 0; i < 4; i++)
							{
								switch((data >> shift) & 3)
								{
									case 0:	*linePtr3 = color00; break;
									case 1:	*linePtr3 = color01; break;
									case 2:	*linePtr3 = color10; break;
									case 3:	*linePtr3 = color11; break;
								}
								
								linePtr3--;
								shift -= 2;
							}
							
							linePtr3 += 8;
							
							for(UInt32 i = 0; i < 4; i++)
							{
								switch((data >> shift) & 3)
								{
									case 0:	*linePtr4 = color00; break;
									case 1:	*linePtr4 = color01; break;
									case 2:	*linePtr4 = color10; break;
									case 3:	*linePtr4 = color11; break;
								}
								
								linePtr4--;
								shift -= 2;
							}
							
							linePtr4 += 8;
						}
					}
				}
			}
		}
		break;
	}
	
	DisposePtr((Ptr)buf);
}

void MTTXMPHandler::DrawToPort(GWorldPtr theWorld, UInt32 xOffset, UInt32 yOffset)
{
	UInt8		* pixBaseAddr = (UInt8 *)GetPixBaseAddr(GetGWorldPixMap(theWorld));
	UInt32		rowBytes = GetPixRowBytes(GetGWorldPixMap(theWorld)) & 0x3FFF;
	
	pixBaseAddr += rowBytes * yOffset;
	pixBaseAddr += (depth / 8) * xOffset;
	
	LockPixels(GetGWorldPixMap(theWorld));
	
	DrawToBuffer(pixBaseAddr, rowBytes, depth);
	
	UnlockPixels(GetGWorldPixMap(theWorld));
}

MTTXMPViewer::MTTXMPViewer()
{
	
}

MTTXMPViewer::~MTTXMPViewer()
{
	
}

void MTTXMPViewer::ParseData(UInt8 * buf, UInt32 size)
{
	handler.AttachFileList(GetOwningFileList());
	
	handler.ParseData(buf, size);
	
	DisposePtr((Ptr)buf);
}

void MTTXMPViewer::GetWindowName(StringPtr theString)
{
	char	buf[256];
	
	if(handler.name.empty())
		std::strcpy(buf, "TXMP: <unnamed>");
	else
		std::sprintf(buf, "TXMP: %s %dx%dx%d", handler.name.c_str(), handler.width, handler.height, handler.depth);
	
	CopyCStringToPascal(buf, theString);
}

void MTTXMPViewer::RecieveMessage(UInt32 messageType, UInt32 messageData)
{
	#pragma unused (messageData)
	
	switch(messageType)
	{
		case 'EXPT':
			if(IsOptionKeyPressed())
			{
				DoExportRaw();
			}
			else
			{
				if(handler.name.empty())
					ExportAsPicture(nil);
				else
					ExportAsPicture(handler.name.c_str());
			}
			break;
		
		case 'INFO':
			SpawnInfoWindow();
			break;
		
		case 'IMPT':
			DoImport();
			break;
	}
}

UInt8 MTTXMPViewer::SupportsMessage(UInt32 messageType, UInt32 messageData)
{
	#pragma unused (messageData)
	
	switch(messageType)
	{
		case 'EXPT':
		case 'INFO':
		case 'IMPT':
			return 1;
	}
	
	return 0;
}

void MTTXMPViewer::DrawToPort(GWorldPtr theWorld)
{
	handler.DrawToPort(theWorld, 0, 0);
}

void MTTXMPViewer::DoExportRaw(void)
{
	MTFileList * fileList = GetOwningFileList();
	
	if(fileList)
		fileList->theFile->ExportSegment(handler.dataOffset, handler.dataSize);
}

void MTTXMPViewer::DoImport(void)
{
	FSSpec					theFile;
	OSErr					theErr;
	GraphicsImportComponent	theImporter = nil;
	PicHandle				thePicture = nil;
	GWorldPtr				theWorld = nil;
	
	try
	{
		if(	((void *)GetGraphicsImporterForFile == (void *)kUnresolvedCFragSymbolAddress) ||
			((void *)GraphicsImportGetAsPicture == (void *)kUnresolvedCFragSymbolAddress))
			throw MTException("QuickTime is needed to import textures.");
		
		if(NavigationGetFileCustom(&theFile, "\pPlease choose the texture you would like to import."))
		{
			theErr = GetGraphicsImporterForFile(&theFile, &theImporter);
			if(theErr)
				throw MTOSException(theErr, "Error importing texture");
			
			theErr = GraphicsImportGetAsPicture(theImporter, &thePicture);
			if(theErr)
				throw MTOSException(theErr, "Error importing texture");
			
			CloseComponent(theImporter);
			theImporter = nil;
			
			HLock((Handle)thePicture);
			
			theErr = NewGWorld(&theWorld, 32, &((*thePicture)->picFrame), nil, nil, 0);
			if(theErr)
				throw MTOSException(theErr, "Error importing texture");
			
			LockPixels(GetGWorldPixMap(theWorld));
			
			{
				MTGWorldSaver	saver(theWorld);
				
				DrawPicture(thePicture, &((*thePicture)->picFrame));
			}
			
			HUnlock((Handle)thePicture);
			ReleaseResource((Handle)thePicture);
			thePicture = nil;
			
			handler.CopyFromGWorld(theWorld);
			
			UnlockPixels(GetGWorldPixMap(theWorld));
			
			DisposeGWorld(theWorld);
		}
	}
	catch(MTException theException)
	{
		theException.DoErrorDialog();
		
		if(theImporter)
		{
			CloseComponent(theImporter);
		}
		if(thePicture)
		{
			HUnlock((Handle)thePicture);
			ReleaseResource((Handle)thePicture);
		}
		if(theWorld)
		{
			DisposeGWorld(theWorld);
		}
	}
	
	UpdateRequest();
}

void MTTXMPViewer::SpawnInfoWindow(void)
{
	MTInfoWindow	* window = new MTInfoWindow;
	char			buf[256];
	
	window->SetOwner(this);
	window->AttachDataTemplate(&kInfoWindowTemplate);
	
	std::sprintf(buf, "%.8X", handler.id);
	window->SetEntryData(0, buf, 0);
	
	std::sprintf(buf, "%.8X", handler.unknown1);
	window->SetEntryData(1, buf, 0);
	
	std::sprintf(buf, "%s", handler.name.c_str());
	window->SetEntryData(2, buf, 0);
	
	std::sprintf(buf, "%.8X", handler.unknown2);
	window->SetEntryData(3, buf, 0);
	
	std::sprintf(buf, "%d", handler.width);
	window->SetEntryData(4, buf, 0);
	
	std::sprintf(buf, "%d", handler.height);
	window->SetEntryData(5, buf, 0);
	
	std::sprintf(buf, "%d", handler.format);
	window->SetEntryData(6, buf, 0);
	
	std::sprintf(buf, "%.8X", handler.unknown4);
	window->SetEntryData(7, buf, 0);
	
	std::sprintf(buf, "%.8X", handler.unknown5);
	window->SetEntryData(8, buf, 0);
	
	std::sprintf(buf, "%.8X", handler.unknown6);
	window->SetEntryData(9, buf, 0);
	
	std::sprintf(buf, "%.8X", handler.dataOffset);
	window->SetEntryData(10, buf, 0);
	
	std::sprintf(buf, "%.8X", handler.unknown7);
	window->SetEntryData(11, buf, 0);
	
	if(handler.name.empty())
		std::sprintf(buf, "TXMP #%d", handler.id);
	else
		std::sprintf(buf, "TXMP %s", handler.name.c_str());
	
	window->SetOwnerName(buf);
	window->RecalculateListSpacing();
	
	gTheApp->AttachWindow(window);
	
	window->HandleUpdateEvent();
}